(function () {
  const PLACEHOLDER_UPLOAD_URL = 'https://passimage.in/i/874c3500329c6adee735.png';
  const canvas = document.getElementById('annot-canvas');
  const ctx = canvas.getContext('2d');
  const canvasWrap = document.getElementById('canvas-wrap');
  const toolbarButtons = Array.from(document.querySelectorAll('#toolbar button[data-tool]'));
  const deleteBtn = document.getElementById('delete-btn');
  const undoBtn = document.getElementById('undo-btn');
  const clearBtn = document.getElementById('clear-btn');
  const downloadBtn = document.getElementById('download-btn');
  const uploadBtn = document.getElementById('upload-btn');
  const copyBtn = document.getElementById('copy-btn');
  const cancelBtn = document.getElementById('cancel-btn');
  const toastEl = document.getElementById('toast');

  const HISTORY = [];
  const HISTORY_LIMIT = 50;

  const STATE = {
    tool: 'rect',
    shapes: [],
    isDrawing: false,
    start: { x: 0, y: 0 },
    preview: null,
    image: null,
    uploadUrl: null,
    tabId: null,
    activeTextInput: null,
    selectedIndex: null,
    dragOrigin: null,
    dragMoved: false,
    dragSnapshot: null
  };

  const STYLE = {
    stroke: '#e53935',
    fill: 'transparent',
    width: 3,
    font: '16px Arial'
  };

  init();

  function init() {
    bindToolbar();
    bindCanvas();
    bindKeyboard();
    chrome.runtime.onMessage.addListener((message) => {
      if (message?.action === 'loadCaptureForAnnotation' && message.imageDataUrl) {
        STATE.uploadUrl = message.placeholderUploadUrl;
        loadImage(message.imageDataUrl);
      }
    });

    chrome.tabs.getCurrent((tab) => {
      STATE.tabId = tab?.id || null;
    });
  }

  function bindToolbar() {
    toolbarButtons.forEach((btn) => {
      btn.addEventListener('click', () => {
        setTool(btn.dataset.tool);
      });
    });
    deleteBtn.addEventListener('click', handleDelete);
    undoBtn.addEventListener('click', handleUndo);
    clearBtn.addEventListener('click', handleClear);
    downloadBtn.addEventListener('click', handleDownload);
    uploadBtn.addEventListener('click', handleUpload);
    copyBtn.addEventListener('click', handleCopy);
    cancelBtn.addEventListener('click', handleCancel);
  }

  function bindCanvas() {
    canvas.addEventListener('mousedown', (e) => {
      if (!STATE.image) return;
      const { x, y } = getCanvasCoords(e);
      if (STATE.tool === 'select') {
        handleSelectDown(x, y);
        return;
      }
      if (STATE.tool === 'text') {
        e.preventDefault();
        e.stopPropagation();
        handleTextPlacement(x, y);
        return;
      }
      STATE.isDrawing = true;
      STATE.start = { x, y };
      STATE.preview = null;
    });

    canvas.addEventListener('mousemove', (e) => {
      if (!STATE.image) return;
      const { x, y } = getCanvasCoords(e);
      if (STATE.tool === 'select' && STATE.dragOrigin) {
        handleSelectMove(x, y);
        return;
      }
      if (!STATE.isDrawing) return;
      STATE.preview = createShape(STATE.tool, STATE.start, { x, y });
      redraw();
    });

    canvas.addEventListener('mouseup', (e) => {
      if (!STATE.image) return;
      if (STATE.tool === 'select' && STATE.dragOrigin) {
        handleSelectUp();
        return;
      }
      if (!STATE.isDrawing) return;
      STATE.isDrawing = false;
      const { x, y } = getCanvasCoords(e);
      const shape = createShape(STATE.tool, STATE.start, { x, y });
      if (shape) {
        pushHistory();
        STATE.shapes.push(shape);
      }
      STATE.preview = null;
      redraw();
    });
  }

  function bindKeyboard() {
    window.addEventListener('keydown', (e) => {
      if (isTyping(e)) return;
      const key = (e.key || '').toLowerCase();

      // Ctrl combos
      if (e.ctrlKey && !e.shiftKey && !e.altKey && !e.metaKey) {
        if (key === 'z') {
          e.preventDefault();
          handleUndo();
          return;
        }
        if (key === 'c') {
          e.preventDefault();
          handleCopy();
          return;
        }
        if (key === 's') {
          e.preventDefault();
          handleDownload();
          return;
        }
        if (key === 'u') {
          e.preventDefault();
          handleUpload();
          return;
        }
        if (key === 'd') {
          e.preventDefault();
          handleCancel();
          return;
        }
        if (key === 'x') {
          e.preventDefault();
          handleClear();
          return;
        }
      }

      // Ignore other modifier combos
      if (e.ctrlKey || e.metaKey || e.altKey) return;

      switch (key) {
        case 'r':
          setTool('rect');
          break;
        case 'c':
          setTool('ellipse');
          break;
        case 't':
          setTool('text');
          break;
        case 'a':
          setTool('arrow');
          break;
        case 's':
          setTool('select');
          break;
        case 'delete':
          handleDelete();
          break;
        default:
          break;
      }
    });
  }

  function loadImage(dataUrl) {
    const img = new Image();
    img.onload = () => {
      STATE.image = img;
      canvas.width = img.width;
      canvas.height = img.height;
      redraw();
    };
    img.onerror = () => showToast('Failed to load image', true);
    img.src = dataUrl;
  }

  function setTool(tool) {
    STATE.tool = tool;
    if (tool !== 'text') {
      removeActiveTextInput();
    }
    if (tool !== 'select') {
      STATE.selectedIndex = null;
      STATE.dragOrigin = null;
    }
    toolbarButtons.forEach((btn) => {
      btn.classList.toggle('active', btn.dataset.tool === tool);
    });
  }

  function isTyping(e) {
    if (STATE.activeTextInput) return true;
    const target = e?.target;
    if (!target) return false;
    const tag = (target.tagName || '').toLowerCase();
    return (
      tag === 'input' ||
      tag === 'textarea' ||
      tag === 'select' ||
      target.isContentEditable
    );
  }

  function getCanvasCoords(e) {
    const rect = canvas.getBoundingClientRect();
    return {
      x: Math.round((e.clientX - rect.left) * (canvas.width / rect.width)),
      y: Math.round((e.clientY - rect.top) * (canvas.height / rect.height))
    };
  }

  function createShape(tool, start, end) {
    const dx = end.x - start.x;
    const dy = end.y - start.y;
    const distSq = dx * dx + dy * dy;
    const minDim = 2; // ignore clicks that don't form a visible shape

    switch (tool) {
      case 'rect':
        if (Math.abs(dx) < minDim || Math.abs(dy) < minDim) return null;
        return {
          type: 'rect',
          x: Math.min(start.x, end.x),
          y: Math.min(start.y, end.y),
          w: Math.abs(end.x - start.x),
          h: Math.abs(end.y - start.y)
        };
      case 'ellipse':
        if (Math.abs(dx) < minDim || Math.abs(dy) < minDim) return null;
        return {
          type: 'ellipse',
          x: Math.min(start.x, end.x),
          y: Math.min(start.y, end.y),
          w: Math.abs(end.x - start.x),
          h: Math.abs(end.y - start.y)
        };
      case 'arrow':
        if (distSq < minDim * minDim) return null;
        return {
          type: 'arrow',
          x1: start.x,
          y1: start.y,
          x2: end.x,
          y2: end.y
        };
      default:
        return null;
    }
  }

  function handleSelectDown(x, y) {
    const idx = hitTestShapes(x, y);
    if (idx === null) {
      STATE.selectedIndex = null;
      STATE.dragOrigin = null;
      STATE.dragSnapshot = null;
      STATE.dragMoved = false;
      redraw();
      return;
    }
    STATE.selectedIndex = idx;
    const shape = STATE.shapes[idx];
    STATE.dragOrigin = {
      pointer: { x, y },
      shape: JSON.parse(JSON.stringify(shape))
    };
    STATE.dragSnapshot = cloneShapes(STATE.shapes);
    STATE.dragMoved = false;
    redraw();
  }

  function handleSelectMove(x, y) {
    if (STATE.selectedIndex === null || !STATE.dragOrigin) return;
    const dx = x - STATE.dragOrigin.pointer.x;
    const dy = y - STATE.dragOrigin.pointer.y;
    const shape = STATE.shapes[STATE.selectedIndex];
    const base = STATE.dragOrigin.shape;
    applyShapeDelta(shape, base, dx, dy);
    STATE.dragMoved = STATE.dragMoved || dx !== 0 || dy !== 0;
    redraw();
  }

  function handleSelectUp() {
    if (STATE.dragMoved && STATE.dragSnapshot) {
      pushHistory(STATE.dragSnapshot);
    }
    STATE.dragOrigin = null;
    STATE.dragSnapshot = null;
    STATE.dragMoved = false;
  }

  function applyShapeDelta(shape, base, dx, dy) {
    switch (shape.type) {
      case 'rect':
      case 'ellipse':
        shape.x = base.x + dx;
        shape.y = base.y + dy;
        break;
      case 'arrow':
        shape.x1 = base.x1 + dx;
        shape.y1 = base.y1 + dy;
        shape.x2 = base.x2 + dx;
        shape.y2 = base.y2 + dy;
        break;
      case 'text':
        shape.x = base.x + dx;
        shape.y = base.y + dy;
        break;
    }
  }

  function handleTextPlacement(x, y) {
    removeActiveTextInput();
    const input = document.createElement('input');
    input.type = 'text';
    input.className = 'text-input';
    positionInputAtCanvasPoint(input, x, y);
    document.body.appendChild(input);
    STATE.activeTextInput = { el: input, x, y, committed: false };
    setTimeout(() => input.focus(), 0);

    const commit = () => {
      if (!STATE.activeTextInput || STATE.activeTextInput.committed) return;
      STATE.activeTextInput.committed = true;
      const text = input.value.trim();
      removeActiveTextInput();
      if (!text) return;
      pushHistory();
      STATE.shapes.push({ type: 'text', x, y, text });
      redraw();
    };
    const cancel = () => {
      removeActiveTextInput();
    };

    input.addEventListener('keydown', (e) => {
      if (e.key === 'Enter') {
        e.preventDefault();
        commit();
      } else if (e.key === 'Escape') {
        e.preventDefault();
        cancel();
      }
    });
    input.addEventListener('blur', () => {
      // Commit on blur to keep flow simple
      commit();
    }, { once: true });
  }

  function positionInputAtCanvasPoint(input, x, y) {
    const rect = canvas.getBoundingClientRect();
    const scaleX = rect.width / canvas.width;
    const scaleY = rect.height / canvas.height;
    const left = rect.left + window.scrollX + x * scaleX;
    const top = rect.top + window.scrollY + y * scaleY;
    input.style.left = `${left}px`;
    input.style.top = `${top}px`;
    input.style.minWidth = '120px';
    input.style.position = 'absolute';
  }

  function removeActiveTextInput() {
    if (!STATE.activeTextInput) return;
    const { el } = STATE.activeTextInput;
    try {
      if (el?.isConnected && typeof el.remove === 'function') {
        el.remove();
      } else if (el?.parentNode) {
        el.parentNode.removeChild(el);
      }
    } catch (e) {
      // ignore removal race
    }
    STATE.activeTextInput = null;
  }

  function redraw() {
    if (!STATE.image) return;
    ctx.clearRect(0, 0, canvas.width, canvas.height);
    ctx.drawImage(STATE.image, 0, 0);
    ctx.lineWidth = STYLE.width;
    ctx.strokeStyle = STYLE.stroke;
    ctx.fillStyle = STYLE.fill;
    ctx.font = STYLE.font;
    ctx.textBaseline = 'top';
    ctx.fillStyle = STYLE.stroke;

    const drawShape = (shape) => {
      switch (shape.type) {
        case 'rect':
          ctx.strokeStyle = STYLE.stroke;
          ctx.fillStyle = STYLE.fill;
          ctx.strokeRect(shape.x, shape.y, shape.w, shape.h);
          ctx.fillRect(shape.x, shape.y, shape.w, shape.h);
          break;
        case 'ellipse':
          ctx.strokeStyle = STYLE.stroke;
          ctx.fillStyle = STYLE.fill;
          ctx.beginPath();
          ctx.ellipse(
            shape.x + shape.w / 2,
            shape.y + shape.h / 2,
            Math.abs(shape.w / 2),
            Math.abs(shape.h / 2),
            0,
            0,
            Math.PI * 2
          );
          ctx.fill();
          ctx.stroke();
          break;
        case 'arrow':
          drawArrow(shape.x1, shape.y1, shape.x2, shape.y2);
          break;
        case 'text':
          ctx.fillStyle = STYLE.stroke;
          ctx.fillText(shape.text, shape.x, shape.y);
          break;
      }
    };

    STATE.shapes.forEach((shape, idx) => {
      drawShape(shape);
      if (STATE.selectedIndex === idx) {
        drawSelectionOutline(shape);
      }
    });
    if (STATE.preview) {
      ctx.save();
      ctx.globalAlpha = 0.7;
      drawShape(STATE.preview);
      ctx.restore();
    }
  }

  function drawArrow(x1, y1, x2, y2) {
    ctx.strokeStyle = STYLE.stroke;
    ctx.fillStyle = STYLE.stroke;
    const headlen = 12;
    const angle = Math.atan2(y2 - y1, x2 - x1);
    ctx.beginPath();
    ctx.moveTo(x1, y1);
    ctx.lineTo(x2, y2);
    ctx.stroke();

    ctx.beginPath();
    ctx.moveTo(x2, y2);
    ctx.lineTo(x2 - headlen * Math.cos(angle - Math.PI / 6), y2 - headlen * Math.sin(angle - Math.PI / 6));
    ctx.lineTo(x2 - headlen * Math.cos(angle + Math.PI / 6), y2 - headlen * Math.sin(angle + Math.PI / 6));
    ctx.closePath();
    ctx.fill();
  }

  function drawSelectionOutline(shape) {
    ctx.save();
    ctx.setLineDash([6, 4]);
    ctx.strokeStyle = '#ffc107';
    switch (shape.type) {
      case 'rect':
      case 'ellipse': {
        const pad = 4;
        ctx.strokeRect(shape.x - pad, shape.y - pad, shape.w + pad * 2, shape.h + pad * 2);
        break;
      }
      case 'text': {
        const pad = 4;
        const width = ctx.measureText(shape.text || '').width || 10;
        const height = parseInt(STYLE.font, 10) || 16;
        ctx.strokeRect(shape.x - pad, shape.y - pad, width + pad * 2, height + pad * 2);
        break;
      }
      case 'arrow': {
        const pad = 6;
        const minX = Math.min(shape.x1, shape.x2) - pad;
        const minY = Math.min(shape.y1, shape.y2) - pad;
        const maxX = Math.max(shape.x1, shape.x2) + pad;
        const maxY = Math.max(shape.y1, shape.y2) + pad;
        ctx.strokeRect(minX, minY, maxX - minX, maxY - minY);
        break;
      }
    }
    ctx.restore();
  }

  function hitTestShapes(x, y) {
    // iterate from top-most to bottom-most
    for (let i = STATE.shapes.length - 1; i >= 0; i--) {
      if (hitTestShape(STATE.shapes[i], x, y)) {
        return i;
      }
    }
    return null;
  }

  function hitTestShape(shape, x, y) {
    const pad = 6;
    switch (shape.type) {
      case 'rect':
      case 'ellipse': {
        const withinX = x >= shape.x - pad && x <= shape.x + shape.w + pad;
        const withinY = y >= shape.y - pad && y <= shape.y + shape.h + pad;
        return withinX && withinY;
      }
      case 'text': {
        ctx.font = STYLE.font;
        const width = ctx.measureText(shape.text || '').width || 10;
        const height = parseInt(STYLE.font, 10) || 16;
        return (
          x >= shape.x - pad &&
          x <= shape.x + width + pad &&
          y >= shape.y - pad &&
          y <= shape.y + height + pad
        );
      }
      case 'arrow': {
        const dist = pointToSegmentDistance(
          x,
          y,
          shape.x1,
          shape.y1,
          shape.x2,
          shape.y2
        );
        return dist <= pad;
      }
      default:
        return false;
    }
  }

  function pointToSegmentDistance(px, py, x1, y1, x2, y2) {
    const dx = x2 - x1;
    const dy = y2 - y1;
    const lenSq = dx * dx + dy * dy;
    if (lenSq === 0) return Math.hypot(px - x1, py - y1);
    let t = ((px - x1) * dx + (py - y1) * dy) / lenSq;
    t = Math.max(0, Math.min(1, t));
    const projX = x1 + t * dx;
    const projY = y1 + t * dy;
    return Math.hypot(px - projX, py - projY);
  }

  function handleUndo() {
    if (!HISTORY.length) return;
    const prev = HISTORY.pop();
    STATE.shapes = cloneShapes(prev);
    STATE.selectedIndex = null;
    STATE.dragOrigin = null;
    STATE.dragSnapshot = null;
    STATE.dragMoved = false;
    STATE.preview = null;
    redraw();
  }

  function handleDelete() {
    if (STATE.selectedIndex === null) {
      return;
    }
    pushHistory();
    STATE.shapes.splice(STATE.selectedIndex, 1);
    STATE.selectedIndex = null;
    STATE.dragOrigin = null;
    redraw();
  }

  function handleClear() {
    pushHistory();
    STATE.shapes = [];
    redraw();
  }

  async function handleCopy() {
    try {
      await copyCanvasToClipboard();
      showToast('Copied to clipboard');
      // await closeTabAfterDelay(); // do not close after copy
    } catch (err) {
      console.error(err);
      showToast('Copy failed', true);
    }
  }

  async function handleUpload() {
    try {
      const blob = await getCanvasBlob();
      const url = await uploadImage(blob);
      await navigator.clipboard.writeText(url);
      await openTab(url, false);
      showToast('Uploaded & link copied');
    } catch (err) {
      console.error(err);
      showToast('Upload/link open failed', true);
    }
  }

  async function handleCancel() {
    await closeTab();
  }

  function copyCanvasToClipboard() {
    return new Promise((resolve, reject) => {
      canvas.toBlob(async (blob) => {
        if (!blob) {
          reject(new Error('Blob not created'));
          return;
        }
        try {
          await navigator.clipboard.write([new ClipboardItem({ 'image/png': blob })]);
          resolve();
        } catch (err) {
          reject(err);
        }
      }, 'image/png');
    });
  }

  async function handleDownload() {
    try {
      const blob = await getCanvasBlob();
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      // set image name to passimage_capture_<timestamp>
      a.download = `passimage_capture_${Date.now()}.png`;
      document.body.appendChild(a);
      a.click();
      setTimeout(() => {
        URL.revokeObjectURL(url);
        a.remove();
      }, 0);
      showToast('Downloaded PNG');
    } catch (err) {
      console.error(err);
      showToast('Download failed', true);
    }
  }

  function openTab(url, active = true) {
    return chrome.tabs.create({ url, active });
  }

  async function closeTab() {
    if (STATE.tabId) {
      try {
        await chrome.tabs.remove(STATE.tabId);
        return;
      } catch (_) {
        // fallback below
      }
    }
    window.close();
  }

  async function closeTabAfterDelay(delayMs = 150) {
    // Give clipboard writes a moment to settle before closing the tab
    await new Promise((resolve) => setTimeout(resolve, delayMs));
    await closeTab();
  }

  function cloneShapes(shapes) {
    return JSON.parse(JSON.stringify(shapes || []));
  }

  function pushHistory(snapshot) {
    const snap = snapshot ? cloneShapes(snapshot) : cloneShapes(STATE.shapes);
    HISTORY.push(snap);
    if (HISTORY.length > HISTORY_LIMIT) {
      HISTORY.shift();
    }
  }

  function getCanvasBlob() {
    return new Promise((resolve, reject) => {
      canvas.toBlob((blob) => {
        if (!blob) {
          reject(new Error('Failed to render image'));
          return;
        }
        resolve(blob);
      }, 'image/png');
    });
  }

  async function uploadImage(blob) {
    if (!blob) throw new Error('Missing image blob');
    if (typeof CONFIG === 'undefined' || !CONFIG.API_URL) {
      throw new Error('Upload config missing');
    }
    const formData = new FormData();
    formData.append('imagedata', blob, 'screenshot.png');

    const headers = {};
    if (CONFIG.API_KEY) {
      headers['X-API-Key'] = CONFIG.API_KEY;
    }

    const resp = await fetch(CONFIG.API_URL, {
      method: 'POST',
      headers,
      body: formData
    });

    const text = (await resp.text()).trim();
    if (!resp.ok) {
      throw new Error(text || `Upload failed (${resp.status})`);
    }
    if (!/^https?:\/\//i.test(text)) {
      throw new Error(text || 'Upload failed');
    }
    return text;
  }

  function cloneShapes(shapes) {
    return JSON.parse(JSON.stringify(shapes || []));
  }

  function pushHistory(snapshot) {
    const snap = snapshot ? cloneShapes(snapshot) : cloneShapes(STATE.shapes);
    HISTORY.push(snap);
    if (HISTORY.length > HISTORY_LIMIT) {
      HISTORY.shift();
    }
  }

  function showToast(message, isError = false) {
    toastEl.textContent = message;
    toastEl.style.background = isError ? 'rgba(217, 48, 37, 0.95)' : 'rgba(32, 32, 32, 0.95)';
    toastEl.classList.add('show');
    setTimeout(() => toastEl.classList.remove('show'), 1700);
  }
})();

