// background.js - Service worker for AS_QUICK_SNAP extension
// Implements region capture, upload, and clipboard/link handling

importScripts('config.js');

const PLACEHOLDER_UPLOAD_URL = 'https://passimage.in/i/874c3500329c6adee735.png';

const logStep = (label, data) => {
  if (data !== undefined) {
    console.log(`[ASQS] ${label}`, data);
  } else {
    console.log(`[ASQS] ${label}`);
  }
};

const sleep = (ms) => new Promise((resolve) => setTimeout(resolve, ms));

const isInjectableUrl = (url) => {
  if (!url) return false;
  // Chrome blocks content scripts on internal pages (chrome://, edge://, etc.)
  // Limit injection to standard web pages to avoid runtime errors.
  return /^https?:\/\//i.test(url);
};

// Trigger overlay injection on icon click
chrome.action.onClicked.addListener(async (tab) => {
  try {
    logStep('Action clicked - injecting content script', { tabId: tab?.id, url: tab?.url });

    if (!isInjectableUrl(tab?.url)) {
      logStep('Injection blocked: unsupported URL', { url: tab?.url });
      showNotification('Open a normal webpage to capture', 'error');
      return;
    }

    await chrome.scripting.executeScript({
      target: { tabId: tab.id },
      files: ['content-script.js']
    });
  } catch (error) {
    console.error('Failed to inject content script:', error);
    showNotification('Failed to start capture', 'error');
  }
});

// Listen for capture requests from the content script
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message?.action === 'captureRegion') {
    logStep('Message: captureRegion', {
      tabId: sender.tab?.id,
      rect: message.rect,
      dpr: message.devicePixelRatio
    });
    handleCapture(sender.tab?.id, message.rect, message.devicePixelRatio)
      .then(() => sendResponse({ success: true }))
      .catch((err) => {
        console.error('Capture failed:', err);
        showNotification('Capture failed', 'error');
        sendResponse({ success: false, error: err.message });
      });
    return true; // Keep channel open for async response
  }

  if (message?.action === 'captureCancelled') {
    showNotification('Capture cancelled', 'info');
    logStep('Message: captureCancelled');
  }

  return false;
});

async function handleCapture(tabId, rect, devicePixelRatio = 1) {
  if (!tabId) {
    throw new Error('Tab not found for capture');
  }
  if (!rect || rect.width <= 0 || rect.height <= 0) {
    throw new Error('Invalid selection');
  }

  showNotification('Capturing...', 'info');

  // Give the content script a moment to remove its overlay before capture
  await sleep(80);

  const dataUrl = await chrome.tabs.captureVisibleTab({ format: 'png' });

  // Adjust coordinates for device pixel ratio so cropping matches overlay
  const scale = devicePixelRatio || 1;
  const scaledRect = {
    x: Math.round(rect.x * scale),
    y: Math.round(rect.y * scale),
    width: Math.round(rect.width * scale),
    height: Math.round(rect.height * scale)
  };

  logStep('Captured visible tab', { dataUrlLength: dataUrl?.length, scale, scaledRect });

  const croppedBlob = await cropImage(dataUrl, scaledRect);
  logStep('Cropped blob ready', { size: croppedBlob.size });

  // Phase 2: hand off to annotation UI (upload temporarily bypassed)
  const croppedDataUrl = await blobToDataUrl(croppedBlob);
  logStep('Converted cropped blob to data URL for annotation', { length: croppedDataUrl.length });

  await openAnnotationTab(croppedDataUrl);
  showNotification('Screenshot ready for annotation', 'info');
}

async function cropImage(dataUrl, rect) {
  const response = await fetch(dataUrl);
  const sourceBlob = await response.blob();
  const imageBitmap = await createImageBitmap(sourceBlob);

  const canvas = new OffscreenCanvas(rect.width, rect.height);
  const ctx = canvas.getContext('2d');
  ctx.drawImage(
    imageBitmap,
    rect.x, rect.y, rect.width, rect.height,
    0, 0, rect.width, rect.height
  );

  return canvas.convertToBlob({ type: 'image/png' });
}

async function blobToDataUrl(blob) {
  return new Promise((resolve, reject) => {
    const reader = new FileReader();
    reader.onloadend = () => resolve(reader.result);
    reader.onerror = (err) => reject(err);
    reader.readAsDataURL(blob);
  });
}

async function uploadImage(blob) {
  const formData = new FormData();
  formData.append('imagedata', blob, 'screenshot.png');

  const response = await fetch(CONFIG.API_URL, {
    method: 'POST',
    headers: CONFIG.API_KEY ? { 'X-API-Key': CONFIG.API_KEY } : undefined,
    body: formData
  });

  const responseText = (await response.text()).trim();

  if (!response.ok) {
    throw new Error(responseText || `Upload failed (${response.status})`);
  }

  // Server returns the absolute URL as plain text on success
  if (!/^https?:\/\//i.test(responseText)) {
    throw new Error(responseText || 'Upload failed');
  }

  return responseText;
}

async function copyTextToClipboard(tabId, text, toastMessage) {
  if (!text) {
    throw new Error('Missing text to copy');
  }

  logStep('Sending text to tab for clipboard copy', { tabId });
  await chrome.tabs.sendMessage(tabId, {
    action: 'copyTextToClipboard',
    text,
    toastMessage,
    toastType: 'success'
  });
}

async function sendImageToTabForClipboard(tabId, dataUrl, toastMessage, toastType = 'success') {
  if (!dataUrl) {
    throw new Error('Missing data URL for clipboard');
  }

  logStep('Sending image to tab for clipboard', { tabId });
  await chrome.tabs.sendMessage(tabId, {
    action: 'copyImageToClipboard',
    dataUrl,
    toastMessage,
    toastType
  });
}

async function openUrlInBackground(url) {
  if (!url) return;
  try {
    await chrome.tabs.create({ url, active: false });
  } catch (error) {
    console.error('Failed to open uploaded image tab:', error);
  }
}

async function openAnnotationTab(imageDataUrl) {
  if (!imageDataUrl) {
    throw new Error('Missing image data for annotation');
  }

  const annotationUrl = chrome.runtime.getURL('annotate.html');
  const tab = await chrome.tabs.create({ url: annotationUrl, active: true });
  await waitForTabComplete(tab.id);

  await chrome.tabs.sendMessage(tab.id, {
    action: 'loadCaptureForAnnotation',
    imageDataUrl,
    placeholderUploadUrl: PLACEHOLDER_UPLOAD_URL
  });
}

function waitForTabComplete(tabId) {
  return new Promise((resolve, reject) => {
    const timeout = setTimeout(() => {
      chrome.tabs.onUpdated.removeListener(listener);
      reject(new Error('Annotation tab did not finish loading'));
    }, 8000);

    const listener = (updatedTabId, info) => {
      if (updatedTabId === tabId && info.status === 'complete') {
        clearTimeout(timeout);
        chrome.tabs.onUpdated.removeListener(listener);
        resolve();
      }
    };

    chrome.tabs.onUpdated.addListener(listener);
  });
}

function showNotification(message, type = 'info') {
  console.log(`[AS_QUICK_SNAP ${type.toUpperCase()}] ${message}`);

  const badgeText = type === 'success' ? '✓' : type === 'error' ? '!' : '...';
  const badgeColor = type === 'success'
    ? '#34a853'
    : type === 'error'
      ? '#d93025'
      : '#4285f4';

  chrome.action.setBadgeText({ text: badgeText });
  chrome.action.setBadgeBackgroundColor({ color: badgeColor });

  setTimeout(() => chrome.action.setBadgeText({ text: '' }), 2000);
}
