// content-script.js - Selection overlay and capture logic

(function () {
  // Avoid re-binding listeners on repeated injections
  if (!window.__ASQS_MESSAGE_BOUND) {
    chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
      if (message?.action === 'copyImageToClipboard' && message.dataUrl) {
        copyImageDataUrlToClipboard(message.dataUrl)
          .then(() => {
            console.log('[ASQS] Image copied to clipboard (content script)');
            showCaptureToast(message.toastMessage || 'Screenshot captured', message.toastType || 'success');
            sendResponse({ success: true });
          })
          .catch((err) => {
            console.error('[ASQS] Clipboard copy failed in content script:', err);
            showCaptureToast('Failed to copy screenshot', 'error');
            sendResponse({ success: false, error: err?.message });
          });
        return true;
      }

      if (message?.action === 'copyTextToClipboard' && message.text) {
        navigator.clipboard.writeText(message.text)
          .then(() => {
            console.log('[ASQS] URL copied to clipboard (content script)');
            showCaptureToast(message.toastMessage || 'Link copied', message.toastType || 'success');
            sendResponse({ success: true });
          })
          .catch((err) => {
            console.error('[ASQS] Clipboard text copy failed in content script:', err);
            showCaptureToast('Failed to copy link', 'error');
            sendResponse({ success: false, error: err?.message });
          });
        return true;
      }

      if (message?.action === 'showToast' && message.message) {
        showCaptureToast(message.message, message.toastType || 'info');
        sendResponse({ success: true });
        return false;
      }
      return false;
    });
    window.__ASQS_MESSAGE_BOUND = true;
  }

  async function copyImageDataUrlToClipboard(dataUrl) {
    const res = await fetch(dataUrl);
    const blob = await res.blob();
    await navigator.clipboard.write([
      new ClipboardItem({ [blob.type || 'image/png']: blob })
    ]);
  }

  class ScreenCaptureOverlay {
  constructor() {
    this.startX = 0;
    this.startY = 0;
    this.isSelecting = false;
    this.selectionBox = null;
    this.overlay = null;
    this.keydownHandler = null;
    this.mouseUpLogged = false;
  }

  init() {
    console.log('[ASQS] Overlay init');
    this.createOverlay();
    this.attachEventListeners();
  }

  createOverlay() {
    this.overlay = document.createElement('div');
    this.overlay.id = 'as-quick-snap-overlay';
    this.overlay.style.cssText = `
      position: fixed;
      top: 0;
      left: 0;
      width: 100%;
      height: 100%;
      background: rgba(0, 0, 0, 0.25);
      z-index: 999999;
      cursor: crosshair;
      user-select: none;
    `;
    document.body.appendChild(this.overlay);

    this.selectionBox = document.createElement('div');
    this.selectionBox.style.cssText = `
      position: absolute;
      border: 2px dashed #e53935;
      background: rgba(229, 57, 53, 0.12);
      pointer-events: none;
      display: none;
    `;
    this.overlay.appendChild(this.selectionBox);
  }

  attachEventListeners() {
    this.overlay.addEventListener('mousedown', (e) => {
      e.preventDefault();
      console.log('[ASQS] mousedown', { x: e.clientX, y: e.clientY });
      this.startX = e.clientX;
      this.startY = e.clientY;
      this.isSelecting = true;
      this.selectionBox.style.display = 'block';
      this.updateSelectionBox(e);
    });

    this.overlay.addEventListener('mousemove', (e) => {
      if (this.isSelecting) {
        if (!this.mouseUpLogged) {
          console.log('[ASQS] mousemove selecting');
          this.mouseUpLogged = true; // avoid flooding logs
        }
        this.updateSelectionBox(e);
      }
    });

    this.overlay.addEventListener('mouseup', (e) => {
      if (!this.isSelecting) return;
      this.isSelecting = false;
      this.updateSelectionBox(e);
      const rect = this.getSelectionRect();
      console.log('[ASQS] mouseup', rect);
      this.cleanup();
      if (rect.width > 2 && rect.height > 2) {
        this.captureRegion(rect);
      } else {
        chrome.runtime.sendMessage({ action: 'captureCancelled' });
      }
    });

    this.keydownHandler = (e) => {
      if (e.key === 'Escape') {
        console.log('[ASQS] Escape pressed - cancelling capture');
        this.cleanup();
        chrome.runtime.sendMessage({ action: 'captureCancelled' });
      }
    };
    document.addEventListener('keydown', this.keydownHandler, true);
  }

  updateSelectionBox(e) {
    const left = Math.min(this.startX, e.clientX);
    const top = Math.min(this.startY, e.clientY);
    const width = Math.abs(e.clientX - this.startX);
    const height = Math.abs(e.clientY - this.startY);

    this.selectionBox.style.left = `${left}px`;
    this.selectionBox.style.top = `${top}px`;
    this.selectionBox.style.width = `${width}px`;
    this.selectionBox.style.height = `${height}px`;
  }

  getSelectionRect() {
    return {
      x: parseInt(this.selectionBox.style.left || '0', 10),
      y: parseInt(this.selectionBox.style.top || '0', 10),
      width: parseInt(this.selectionBox.style.width || '0', 10),
      height: parseInt(this.selectionBox.style.height || '0', 10)
    };
  }

  cleanup() {
    if (this.overlay && this.overlay.parentNode) {
      this.overlay.parentNode.removeChild(this.overlay);
    }
    if (this.keydownHandler) {
      document.removeEventListener('keydown', this.keydownHandler, true);
    }
  }

  async captureRegion(rect) {
    try {
      console.log('[ASQS] Sending captureRegion request', rect);
      await chrome.runtime.sendMessage({
        action: 'captureRegion',
        rect,
        devicePixelRatio: window.devicePixelRatio || 1
      });
    } catch (error) {
      console.error('Failed to request capture:', error);
    }
  }
  }

  // Initialize overlay when script is injected
  const overlay = new ScreenCaptureOverlay();
  overlay.init();

  function showCaptureToast(message, type = 'info') {
    const colors = {
      success: 'rgba(52, 168, 83, 0.95)',
      error: 'rgba(217, 48, 37, 0.95)',
      info: 'rgba(66, 133, 244, 0.95)'
    };
    const background = colors[type] || colors.info;

    // Remove any existing toast first
    const existing = document.getElementById('asqs-toast');
    if (existing) existing.remove();

    const toast = document.createElement('div');
    toast.id = 'asqs-toast';
    toast.textContent = message;
    toast.style.cssText = `
      position: fixed;
      top: 16px;
      left: 50%;
      padding: 10px 14px;
      background: ${background};
      color: #fff;
      border-radius: 6px;
      box-shadow: 0 4px 12px rgba(0,0,0,0.25);
      font-size: 13px;
      font-family: Arial, sans-serif;
      z-index: 1000000;
      opacity: 0;
      transition: opacity 150ms ease-out, transform 150ms ease-out;
      transform: translate(-50%, -6px);
    `;
    document.body.appendChild(toast);

    requestAnimationFrame(() => {
      toast.style.opacity = '1';
      toast.style.transform = 'translate(-50%, 0)';
    });

    setTimeout(() => {
      toast.style.opacity = '0';
      toast.style.transform = 'translate(-50%, -6px)';
      setTimeout(() => toast.remove(), 200);
    }, 2000);
  }
})();
